package com.proxyclient.client;

import com.proxyclient.client.model.FlowAccountInfo;
import com.proxyclient.client.model.PackageInfo;
import com.proxyclient.client.model.FlowDetailInfo;

import java.io.*;
import java.net.*;
import java.util.*;
import java.util.stream.Collectors;
import java.lang.reflect.Type;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

// 导入Gson库
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.reflect.TypeToken;

/**
 * proxyclient API客户端
 */
public class ProxyClient {
    private static final String DEFAULT_BASE_URL = "http://smartcheapip.com/api/v1/";
    private static final int DEFAULT_TIMEOUT = 30000; // 30秒
    
    private String baseUrl;
    private String apiKey;
    private int timeout;
    private Gson gson;
    
    /**
     * 客户端配置
     */
    public static class Config {
        private String apiKey;
        private String baseUrl = DEFAULT_BASE_URL;
        private int timeout = DEFAULT_TIMEOUT;
        
        public Config(String apiKey) {
            this.apiKey = apiKey;
        }
        
        public Config setBaseUrl(String baseUrl) {
            this.baseUrl = baseUrl;
            return this;
        }
        
        public Config setTimeout(int timeout) {
            this.timeout = timeout;
            return this;
        }
    }
    
    /**
     * API响应结构
     */
    private static class ApiResponse {
        private int code;
        private JsonElement data;
        private String msg;
        
        // Getter和Setter方法
        public int getCode() { return code; }
        public void setCode(int code) { this.code = code; }
        
        public JsonElement getData() { return data; }
        public void setData(JsonElement data) { this.data = data; }
        
        public String getMsg() { return msg; }
        public void setMsg(String msg) { this.msg = msg; }
    }
    
    /**
     * 创建新的API客户端
     * @param config 客户端配置
     * @throws ApiException 如果缺少apikey会抛出异常
     */
    public ProxyClient(Config config) throws ApiException {
        if (config.apiKey == null || config.apiKey.trim().isEmpty()) {
            throw new ApiException("需要传入apikey");
        }
        
        this.apiKey = config.apiKey;
        this.baseUrl = config.baseUrl != null ? config.baseUrl : DEFAULT_BASE_URL;
        this.timeout = config.timeout > 0 ? config.timeout : DEFAULT_TIMEOUT;
        
        // 初始化Gson
        this.gson = new GsonBuilder()
                .setDateFormat("yyyy-MM-dd HH:mm:ss")
                .create();
    }
    
    /**
     * 向服务器发送请求
     * @param method HTTP方法
     * @param uri 请求URI
     * @param data 请求数据
     * @return 响应数据
     * @throws ApiException 请求失败时抛出异常
     */
    private Object request(String method, String uri, Map<String, Object> data) throws ApiException {
        if (data == null) {
            data = new HashMap<>();
        }
        data.put("apikey", this.apiKey);
        
        try {
            String urlStr = baseUrl + uri;
            URL url = new URL(urlStr);
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            
            // 设置连接参数
            connection.setConnectTimeout(timeout);
            connection.setReadTimeout(timeout);
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            connection.setRequestProperty("Accept-Encoding", "gzip, deflate");
            connection.setRequestProperty("User-Agent", "proxyclient-Java-Client/1.0");
            
            // 构建请求参数
            String postData = buildPostData(data);
            
            // 发送请求
            try (DataOutputStream wr = new DataOutputStream(connection.getOutputStream())) {
                wr.writeBytes(postData);
                wr.flush();
            }
            
            // 读取响应
            int responseCode = connection.getResponseCode();
            // StringBuilder response = new StringBuilder();
            // try (BufferedReader in = new BufferedReader(
            //         new InputStreamReader(connection.getInputStream()))) {
            //     String inputLine;
            //     while ((inputLine = in.readLine()) != null) {
            //         response.append(inputLine);
            //     }
            // }
            // 读取响应（支持gzip和deflate压缩）
            String response = readCompressedResponse(connection);
            connection.disconnect();
            
            // 解析响应
            return parseResponse(response);
            
        } catch (Exception e) {
            throw new ApiException("请求失败: " + e.getMessage(), e);
        }
    }
     /**
     * 读取压缩的HTTP响应
     */
    private String readCompressedResponse(HttpURLConnection connection) throws IOException {
        int responseCode = connection.getResponseCode();
        
        // 检查响应编码
        String contentEncoding = connection.getContentEncoding();
        InputStream inputStream;
        
        if (responseCode >= 400) {
            // 错误响应
            inputStream = connection.getErrorStream();
            if (inputStream == null) {
                inputStream = connection.getInputStream();
            }
        } else {
            // 正常响应
            inputStream = connection.getInputStream();
        }
        
        if (inputStream == null) {
            throw new IOException("无法获取响应流");
        }
        
        // 根据Content-Encoding选择合适的解压方式
        if (contentEncoding != null) {
            if (contentEncoding.equalsIgnoreCase("gzip")) {
                inputStream = new GZIPInputStream(inputStream);
            } else if (contentEncoding.equalsIgnoreCase("deflate")) {
                inputStream = new InflaterInputStream(inputStream, new Inflater(true));
            }
        }
        
        // 读取响应内容
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"))) {
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }
            return response.toString();
        }
    }
    
    /**
     * 构建POST数据
     */
    private String buildPostData(Map<String, Object> data) {
        return data.entrySet().stream()
                .map(entry -> {
                    try {
                        String key = URLEncoder.encode(entry.getKey(), "UTF-8");
                        String value = URLEncoder.encode(convertToString(entry.getValue()), "UTF-8");
                        return key + "=" + value;
                    } catch (UnsupportedEncodingException e) {
                        return "";
                    }
                })
                .filter(s -> !s.isEmpty())
                .collect(Collectors.joining("&"));
    }
    
    /**
     * 解析API响应 - 使用Gson进行正确的JSON解析
     */
    private Object parseResponse(String response) throws ApiException {
        try {
            // 首先尝试解析为标准的API响应结构
            ApiResponse apiResponse = gson.fromJson(response, ApiResponse.class);
            
            if (apiResponse != null) {
                int code = apiResponse.getCode();
                
                if (code == 0 || code == 200) {
                    // 正常响应
                    JsonElement dataElement = apiResponse.getData();
                    if (dataElement != null && !dataElement.isJsonNull()) {
                        // 返回原始JsonElement，让具体方法决定如何解析
                        return dataElement;
                    } else {
                        return true;
                    }
                } else {
                    // 错误响应
                    String errorMsg = apiResponse.getMsg() != null ? 
                            apiResponse.getMsg() : "request fail";
                    throw new ApiException(errorMsg, code);
                }
            }
            
            // 如果不是标准API响应格式，尝试直接解析
            JsonElement rootElement = JsonParser.parseString(response);
            if (rootElement.isJsonObject()) {
                // 如果是JSON对象，检查是否有code字段
                if (rootElement.getAsJsonObject().has("code")) {
                    int code = rootElement.getAsJsonObject().get("code").getAsInt();
                    if (code == 0 || code == 200) {
                        if (rootElement.getAsJsonObject().has("data")) {
                            return rootElement.getAsJsonObject().get("data");
                        }
                        return true;
                    } else {
                        String errorMsg = rootElement.getAsJsonObject().has("msg") ?
                                rootElement.getAsJsonObject().get("msg").getAsString() : "request fail";
                        throw new ApiException(errorMsg, code);
                    }
                }
            }
            
            // 如果都不是，返回原始内容
            return response;
            
        } catch (ApiException e) {
            throw e;
        } catch (Exception e) {
            // JSON解析失败，返回原始内容
            return response;
        }
    }
    
    /**
     * 将各种类型转换为字符串
     */
    private String convertToString(Object value) {
        if (value == null) {
            return "";
        }
        if (value instanceof String) {
            return (String) value;
        }
        return value.toString();
    }
    
    /**
     * 创建流量线路
     * @param country 国家编码
     * @param ttl 设备在线时间（分钟）
     * @return 创建的流量线路信息列表
     * @throws ApiException 请求失败时抛出异常
     */
    public List<FlowAccountInfo> createAccount(String country, int ttl) throws ApiException {
        Map<String, Object> data = new HashMap<>();
        data.put("country", country);
        data.put("ttl", ttl);
        
        Object response = request("POST", "accounts/create", data);
        
        if (response instanceof JsonElement) {
            JsonElement dataElement = (JsonElement) response;
            if (dataElement.isJsonArray()) {
                // 解析为FlowAccountInfo列表
                Type listType = new TypeToken<List<FlowAccountInfo>>(){}.getType();
                return gson.fromJson(dataElement, listType);
            } else if (dataElement.isJsonObject()) {
                // 如果是单个对象，包装成列表
                FlowAccountInfo account = gson.fromJson(dataElement, FlowAccountInfo.class);
                return Arrays.asList(account);
            }
        } else if (response instanceof String) {
            // 如果是字符串响应，可能是错误信息
            throw new ApiException("创建账户失败: " + response);
        }
        
        throw new ApiException("创建账户响应格式错误");
    }
    
    /**
     * 获取已经创建的流量线路列表
     * @param country 国家编码
     * @param offset 分页的偏移量
     * @param limit 每页的数量（最大100）
     * @return 流量线路信息列表
     * @throws ApiException 请求失败时抛出异常
     */
    public List<FlowAccountInfo> getAccountList(String country, int offset, int limit) throws ApiException {
        Map<String, Object> data = new HashMap<>();
        data.put("country", country != null ? country : "");
        data.put("offset", offset);
        data.put("limit", limit);
        
        Object response = request("POST", "accounts/get_list", data);
        
        if (response instanceof JsonElement) {
            JsonElement dataElement = (JsonElement) response;
            if (dataElement.isJsonArray()) {
                Type listType = new TypeToken<List<FlowAccountInfo>>(){}.getType();
                return gson.fromJson(dataElement, listType);
            }
        } else if (response instanceof String) {
            throw new ApiException("获取账户列表失败: " + response);
        }
        
        throw new ApiException("获取账户列表响应格式错误");
    }
    
    /**
     * 删除流量线路
     * @param account 要删掉的线路账号名
     * @return 是否删除成功
     * @throws ApiException 请求失败时抛出异常
     */
    public boolean removeAccount(String account) throws ApiException {
        Map<String, Object> data = new HashMap<>();
        data.put("account", account);
        
        Object response = request("POST", "accounts/remove", data);
        
        if (response instanceof Boolean) {
            return (Boolean) response;
        } else if (response instanceof JsonElement) {
            JsonElement dataElement = (JsonElement) response;
            if (dataElement.isJsonPrimitive() && dataElement.getAsJsonPrimitive().isBoolean()) {
                return dataElement.getAsBoolean();
            }
        } else if (response instanceof String) {
            // 如果返回字符串"true"也算成功
            return "true".equals(response);
        }
        
        return false;
    }
    
    /**
     * 获取整体流量使用情况
     * @return 流量使用情况
     * @throws ApiException 请求失败时抛出异常
     */
    public PackageInfo getPackageInfo() throws ApiException {
        Object response = request("POST", "accounts/get_package_info", new HashMap<>());
        
        if (response instanceof JsonElement) {
            JsonElement dataElement = (JsonElement) response;
            if (dataElement.isJsonObject()) {
                return gson.fromJson(dataElement, PackageInfo.class);
            }
        } else if (response instanceof String) {
            throw new ApiException("获取套餐信息失败: " + response);
        }
        
        throw new ApiException("获取套餐信息响应格式错误");
    }
    
    /**
     * 查询时间范围之内的流量使用明细
     * @param account 流量线路的账号名（空字符串表示全部）
     * @param startTime 开始时间（格式YYYY-mm-dd HH:ii:ss）
     * @param endTime 结束时间（格式YYYY-mm-dd HH:ii:ss）
     * @return 流量使用明细列表
     * @throws ApiException 请求失败时抛出异常
     */
    public List<FlowDetailInfo> getTrafficDetail(String account, String startTime, String endTime) throws ApiException {
        Map<String, Object> data = new HashMap<>();
        data.put("account", account != null ? account : "");
        data.put("start_time", startTime);
        data.put("end_time", endTime);
        
        Object response = request("POST", "accounts/get_traffic_detail", data);
        
        if (response instanceof JsonElement) {
            JsonElement dataElement = (JsonElement) response;
            if (dataElement.isJsonArray()) {
                Type listType = new TypeToken<List<FlowDetailInfo>>(){}.getType();
                return gson.fromJson(dataElement, listType);
            }
        } else if (response instanceof String) {
            throw new ApiException("获取流量详情失败: " + response);
        }
        
        throw new ApiException("获取流量详情响应格式错误");
    }
    
    /**
     * 便捷方法：获取默认账户列表
     * @return 账户列表
     * @throws ApiException 请求失败时抛出异常
     */
    public List<FlowAccountInfo> getAccountListDefault() throws ApiException {
        return getAccountList("", 0, 10);
    }
    
    /**
     * 便捷方法：按国家获取账户列表
     * @param country 国家编码
     * @return 账户列表
     * @throws ApiException 请求失败时抛出异常
     */
    public List<FlowAccountInfo> getAccountListWithCountry(String country) throws ApiException {
        return getAccountList(country, 0, 100);
    }
}