const { ProxyClient, Config, ApiError } = require('../src');

/**
 * 使用示例
 */
async function main() {
    // 配置客户端
    const config = new Config({
        apiKey: "your_apikey_here",
        timeout: 30000
    });

    try {
        // 创建客户端
        const client = new ProxyClient(config);

        console.log("=== SmartCheapIP Node.js Client Examples ===");

        // 示例1: 创建流量线路
        console.log("\n1. 创建流量线路...");
        try {
            const accounts = await client.createAccount("US", 60);
            console.log(`创建成功，账户数量: ${accounts.length}`);
            accounts.forEach(account => {
                console.log(`账户: ${account.account}, 国家: ${account.country}, 代理IP: ${account.proxy_ip}:${account.s5_port}`);
            });
        } catch (error) {
            console.log(`创建账户失败: ${error.message}`);
        }

        // 示例2: 获取流量线路列表
        console.log("\n2. 获取流量线路列表...");
        try {
            const accountList = await client.getAccountList("US", 0, 10);
            console.log(`获取到 ${accountList.length} 个账户`);
            accountList.forEach(account => {
                console.log(`账户ID: ${account.account_id}, 账号: ${account.account}, 国家: ${account.country}`);
            });
        } catch (error) {
            console.log(`获取账户列表失败: ${error.message}`);
        }

        // 示例3: 获取套餐信息
        console.log("\n3. 获取套餐信息...");
        try {
            const packageInfo = await client.getPackageInfo();
            console.log(`总流量: ${packageInfo.traffic_total} ${packageInfo.unit}`);
            console.log(`已使用: ${packageInfo.traffic_used} ${packageInfo.unit}`);
            console.log(`剩余流量: ${packageInfo.traffic_remain} ${packageInfo.unit}`);
            console.log(`账户数量: ${packageInfo.flow_account_count}`);
        } catch (error) {
            console.log(`获取套餐信息失败: ${error.message}`);
        }

        // 示例4: 查询流量明细
        console.log("\n4. 查询流量明细...");
        try {
            const trafficDetails = await client.getTrafficDetail(
                "", // 空字符串表示查询全部账户
                "2025-11-01 00:00:00",
                "2025-11-30 23:59:59"
            );
            console.log(`查询到 ${trafficDetails.length} 条流量记录`);
            trafficDetails.slice(0, 3).forEach(detail => { // 只显示前3条
                console.log(`日期: ${detail.date}, 账户: ${detail.account}, 流量: ${detail.traffic}`);
            });
        } catch (error) {
            console.log(`查询流量明细失败: ${error.message}`);
        }

        // 示例5: 删除流量线路
        console.log("\n5. 删除流量线路...");
        try {
            const result = await client.removeAccount("test_account_123");
            console.log(`删除结果: ${result ? "成功" : "失败"}`);
        } catch (error) {
            console.log(`删除账户失败: ${error.message}`);
        }



    } catch (error) {
        if (error instanceof ApiError) {
            console.error(`API错误 [${error.code}]: ${error.message}`);
        } else {
            console.error(`系统错误: ${error.message}`);
        }
    }
}

/**
 * 错误处理示例
 */
async function handleApiError() {
    const config = new Config({
        apiKey: "invalid_key", // 使用无效的API密钥测试错误处理
    });

    try {
        const client = new SmartCheapIPClient(config);
        await client.getPackageInfo();
    } catch (error) {
        if (error instanceof ApiError) {
            console.log(`API业务错误: 代码=${error.code}, 消息=${error.message}`);
        } else {
            console.log(`网络或其他错误: ${error.message}`);
        }
    }
}

// 运行示例
if (require.main === module) {
    main().catch(console.error);
}

module.exports = {
    main,
    handleApiError
};