const axios = require('axios');
const { FlowAccountInfo, PackageInfo, FlowDetailInfo, ApiError } = require('./types');

/**
 * 客户端配置
 */
class Config {
    /**
     * @param {Object} options
     * @param {string} options.apiKey
     * @param {string} options.baseURL
     * @param {number} options.timeout
     */
    constructor(options = {}) {
        this.apiKey = options.apiKey || '';
        this.baseURL = options.baseURL || 'http://smartcheapip.com/api/v1/';
        this.timeout = options.timeout || 30000; // 默认30秒
    }
}

/**
 * API响应结构
 */
class ApiResponse {
    /**
     * @param {number} code
     * @param {any} data
     * @param {string} msg
     */
    constructor(code, data, msg) {
        this.code = code;
        this.data = data;
        this.msg = msg;
    }
}

/**
 * SmartCheapIP API客户端
 */
class ProxyClient {
    /**
     * 创建新的API客户端
     * @param {Config} config 客户端配置
     * @throws {ApiError} 如果缺少apikey会抛出异常
     */
    constructor(config) {
        if (!config || !config.apiKey) {
            throw new ApiError('需要传入apikey');
        }

        this.config = config;
        
        // 创建axios实例
        this.httpClient = axios.create({
            baseURL: config.baseURL,
            timeout: config.timeout,
            headers: {
                'User-Agent': 'SmartCheapIP-NodeJS-Client/1.0',
                'Content-Type': 'application/x-www-form-urlencoded'
            }
        });

        // 添加请求拦截器
        this.httpClient.interceptors.request.use(
            (config) => {
                // 确保data存在
                if (!config.data) {
                    config.data = {};
                }
                
                // 添加apikey参数
                if (config.data instanceof URLSearchParams) {
                    config.data.append('apikey', this.config.apiKey);
                } else if (typeof config.data === 'object') {
                    config.data.apikey = this.config.apiKey;
                }
                
                return config;
            },
            (error) => {
                return Promise.reject(error);
            }
        );

        // 添加响应拦截器
        this.httpClient.interceptors.response.use(
            (response) => {
                return this.parseResponse(response.data);
            },
            (error) => {
                if (error.response) {
                    // 服务器返回错误状态码
                    throw new ApiError(`HTTP错误: ${error.response.status}`, error.response.status);
                } else if (error.request) {
                    // 请求发送失败
                    throw new ApiError('网络请求失败: ' + error.message);
                } else {
                    // 其他错误
                    throw new ApiError('请求配置错误: ' + error.message);
                }
            }
        );
    }

    /**
     * 解析API响应
     * @param {any} response 原始响应
     * @returns {Promise<any>} 解析后的数据
     * @throws {ApiError} API错误时抛出异常
     */
    async parseResponse(response) {
        // 如果响应是字符串，尝试解析JSON
        if (typeof response === 'string') {
            try {
                response = JSON.parse(response);
            } catch (error) {
                // 如果不是JSON，返回原始内容
                return response;
            }
        }

        // 检查响应结构
        if (response && typeof response === 'object' && 'code' in response) {
            if (response.code === 0 || response.code === 200) {
                // 正常响应
                return response.data !== undefined ? response.data : true;
            } else {
                // 错误响应
                throw new ApiError(response.msg || 'request fail', response.code);
            }
        }

        // 未知响应格式，返回原始数据
        return response;
    }

    /**
     * 向服务器发送请求
     * @param {string} method HTTP方法
     * @param {string} uri 请求URI
     * @param {Object} data 请求数据
     * @returns {Promise<any>} 响应数据
     * @throws {ApiError} 请求失败时抛出异常
     */
    async request(method, uri, data = {}) {
        const config = {
            method: method.toLowerCase(),
            url: uri
        };

        if (method.toUpperCase() === 'POST') {
            const formData = new URLSearchParams();
            for (const [key, value] of Object.entries(data)) {
                formData.append(key, this.convertToString(value));
            }
            config.data = formData;
        } else {
            // 对于GET请求，将参数添加到URL
            const params = new URLSearchParams();
            for (const [key, value] of Object.entries(data)) {
                params.append(key, this.convertToString(value));
            }
            config.url = `${uri}?${params.toString()}`;
        }

        return await this.httpClient.request(config);
    }

    /**
     * 将各种类型转换为字符串
     * @param {any} value 要转换的值
     * @returns {string} 转换后的字符串
     */
    convertToString(value) {
        if (value === null || value === undefined) {
            return '';
        }
        
        switch (typeof value) {
            case 'string':
                return value;
            case 'number':
                return value.toString();
            case 'boolean':
                return value ? 'true' : 'false';
            case 'object':
                return JSON.stringify(value);
            default:
                return String(value);
        }
    }

    /**
     * 创建流量线路
     * @param {string} country 国家编码
     * @param {number} ttl 设备在线时间（分钟）
     * @returns {Promise<FlowAccountInfo[]>} 返回创建的流量线路信息数组
     * @throws {ApiError} 请求失败时抛出异常
     */
    async createAccount(country, ttl) {
        const data = {
            country: country,
            ttl: ttl
        };
        
        const response = await this.request('POST', 'accounts/create', data);
        
        if (Array.isArray(response)) {
            return response.map(item => new FlowAccountInfo(item));
        }
        
        throw new ApiError('创建账户响应格式错误');
    }

    /**
     * 获取已经创建的流量线路列表
     * @param {string} country 国家编码
     * @param {number} offset 分页的偏移量
     * @param {number} limit 每页的数量（最大100）
     * @returns {Promise<FlowAccountInfo[]>} 返回获取的流量线路信息列表
     * @throws {ApiError} 请求失败时抛出异常
     */
    async getAccountList(country = '', offset = 0, limit = 10) {
        const data = {
            country: country,
            offset: offset,
            limit: limit
        };
        
        const response = await this.request('POST', 'accounts/get_list', data);
        
        if (Array.isArray(response)) {
            return response.map(item => new FlowAccountInfo(item));
        }
        
        throw new ApiError('获取账户列表响应格式错误');
    }

    /**
     * 删除流量线路
     * @param {string} account 要删掉的线路账号名
     * @returns {Promise<boolean>} 是否删除成功
     * @throws {ApiError} 请求失败时抛出异常
     */
    async removeAccount(account) {
        const data = {
            account: account
        };
        
        const response = await this.request('POST', 'accounts/remove', data);
        return response === true;
    }

    /**
     * 获取整体流量使用情况
     * @returns {Promise<PackageInfo>} 返回获取的流量使用情况
     * @throws {ApiError} 请求失败时抛出异常
     */
    async getPackageInfo() {
        const response = await this.request('POST', 'accounts/get_package_info', {});
        
        if (response && typeof response === 'object') {
            return new PackageInfo(response);
        }
        
        throw new ApiError('获取套餐信息响应格式错误');
    }

    /**
     * 查询时间范围之内的流量使用明细
     * @param {string} account 流量线路的账号名（空字符串表示全部）
     * @param {string} startTime 开始时间（格式YYYY-mm-dd HH:ii:ss）
     * @param {string} endTime 结束时间（格式YYYY-mm-dd HH:ii:ss）
     * @returns {Promise<FlowDetailInfo[]>} 返回查询到的流量使用明细
     * @throws {ApiError} 请求失败时抛出异常
     */
    async getTrafficDetail(account = '', startTime = '', endTime = '') {
        const data = {
            account: account,
            start_time: startTime,
            end_time: endTime
        };
        
        const response = await this.request('POST', 'accounts/get_traffic_detail', data);
        
        if (Array.isArray(response)) {
            return response.map(item => new FlowDetailInfo(item));
        }
        
        throw new ApiError('获取流量详情响应格式错误');
    }

    /**
     * 便捷方法：获取默认账户列表
     * @returns {Promise<FlowAccountInfo[]>} 账户列表
     * @throws {ApiError} 请求失败时抛出异常
     */
    async getAccountListDefault() {
        return await this.getAccountList('', 0, 10);
    }

    /**
     * 便捷方法：按国家获取账户列表
     * @param {string} country 国家编码
     * @returns {Promise<FlowAccountInfo[]>} 账户列表
     * @throws {ApiError} 请求失败时抛出异常
     */
    async getAccountListWithCountry(country) {
        return await this.getAccountList(country, 0, 100);
    }
}

module.exports = {
    ProxyClient,
    Config,
    FlowAccountInfo,
    PackageInfo,
    FlowDetailInfo,
    ApiError
};